<?php

/**
 * Authenticate QuickBooks Web Connector users against a .CSV (comma-separated-value) file
 * 
 * @author Keith Palmer <keith@consolibyte.com>
 * @license LICENSE.txt
 *  
 * @package QuickBooks
 * @subpackage Authenticate
 */

/**
 * Authenticate interface
 */
require_once 'QuickBooks/Authenticate.php';

/**
 * Utilities for parsing DSN string
 */
require_once 'QuickBooks/Utilities.php';

/**
 * 
 */
class QuickBooks_Authenticate_Csv implements QuickBooks_Authenticate
{
	/**
	 * An array mapping usernames to encrypted passwords
	 * @var array
	 */
	protected $_accounts;
	
	/**
	 * Create a new authentication handler
	 * 
	 * @param string $dsn	DSN-style connection string, something like: htpasswd:///path/to/your/htpasswd/file.htpasswd
	 */
	public function __construct($dsn)
	{
		ini_set('auto_detect_line_endings', true);
		
		$conn_defaults = array(
			'scheme' => 'csv', 
			'user' => '', 				// ignored
			'pass' => '',				// ignored
			'host' => '',  				// ignored (assume local file)
			'port' => '',				// ignored
			'path' => '/path/to/file.csv',  
			'query' => '', 				// ignored
			);
		
		// htpasswd:///database?table_name=quickbooks_user&field_username=username&field_password=password&crypt_function=md5
		$parse = QuickBooks_Utilities::parseDSN($dsn, $conn_defaults);
		
		$this->_accounts = array();
		if (is_file($parse['path']) and $fp = fopen($parse['path'], 'r'))
		{
			while (false !== ($arr = fgetcsv($fp, 1000, ',')))
			{
				if (empty($arr[0]) or empty($arr[1]))
				{
					continue;
				}
				
				for ($i = 2; $i < 10; $i++)
				{
					if (empty($arr[$i]))
					{
						$arr[$i] = '';
					}
				}
				
				$username = trim($arr[0]);
				
				$this->_accounts[$username] = $arr;
			}
		}
	}
	
	/**
	 * Authenticate a username and password against an 'htpasswd' file (generated by Apache) 
	 * 
	 * @param string $username
	 * @param string $password
	 * @param string $company_file				The htpasswd authentication method does not this (you can still override it in the quickbooks_user table though)
	 * @param integer $wait_before_next_update	
	 * @param integer $min_run_every_n_seconds
	 * @return boolean
	 */
	public function authenticate($username, $password, &$company_file, &$wait_before_next_update, &$min_run_every_n_seconds)
	{
		if (isset($this->_accounts[$username]))
		{
			$ok = false;
			
			if ($password = $this->_accounts[$username][1])		// plain text password, OK!
			{
				$ok = true;
			}
			else if (md5($password) == $this->_accounts[$username][1])		// md5 hashed passwords, OK!
			{
				$ok = true;
			}
			else if (sha1($password) == $this->_accounts[$username][1])		// sha1 passwords
			{
				$ok = true;
			}
			else if (crypt($password) == $this->_accounts[$username][1])	// crypt() passwords
			{
				$ok = true;
			}
			
			if ($ok)
			{
				$company_file = $this->_accounts[$username][2];
				$wait_before_next_update = (int) $this->_accounts[$username][3];
				$min_run_every_n_seconds = (int) $this->_accounts[$username][4];
				
				return true;
			}
		}
		
		return false;
	}
}
