<?php

/**
 * Authenticate against 'htpasswd' files which have been generated by the Apache web server 
 * 
 * The following authentication schemes are supported:
 * 	- CRYPT		(default for the 'htpasswd' utility)
 * 	- APR1-MD5	(modified MD5 hash algorithm by the Apache project)
 * 	- SHA1		(Hex packed and base64 encoded)
 * 
 * @author Keith Palmer <keith@consolibyte.com>
 * @license LICENSE.txt
 * 
 * @package QuickBooks
 * @subpackage Authenticate
 */

/**
 * Authenticate interface
 */
require_once 'QuickBooks/Authenticate.php';

/**
 * Utilities for parsing DSN string
 */
require_once 'QuickBooks/Utilities.php';

/**
 * Authenticate QuickBooks Web Connector users via Apache htpasswd files
 */
class QuickBooks_Authenticate_Htpasswd implements QuickBooks_Authenticate
{
	/**
	 * An array mapping usernames to encrypted passwords
	 * @var array
	 */
	protected $_accounts;
	
	/**
	 * Create a new authentication handler
	 * 
	 * @param string $dsn	DSN-style connection string, something like: htpasswd:///path/to/your/htpasswd/file.htpasswd
	 */
	public function __construct($dsn)
	{
		$conn_defaults = array(
			'scheme' => 'htpasswd', 
			'user' => '', 				// ignored
			'pass' => '',				// ignored
			'host' => 'localhost',  	// ignored (assume local file)
			'port' => 3306,				// ignored
			'path' => '/path/to/htpasswd',  
			'query' => '', 				// ignored
			);
		
		// htpasswd:///database?table_name=quickbooks_user&field_username=username&field_password=password&crypt_function=md5
		$parse = QuickBooks_Utilities::parseDSN($dsn, $conn_defaults);
		
		$this->_accounts = array();
		if (is_file($parse['path']) and $lines = file($parse['path']))
		{
			foreach ($lines as $line)
			{
				$explode = explode(':', trim($line));
				
				$this->_accounts[$explode[0]] = $explode[1];
			}
		}
	}
	
	/**
	 * Authenticate a username and password against an 'htpasswd' file (generated by Apache) 
	 * 
	 * @param string $username
	 * @param string $password
	 * @param string $company_file				The htpasswd authentication method does not this (you can still override it in the quickbooks_user table though)
	 * @param integer $wait_before_next_update	
	 * @param integer $min_run_every_n_seconds
	 * @return boolean
	 */
	public function authenticate($username, $password, &$company_file, &$wait_before_next_update, &$min_run_every_n_seconds)
	{
		$func = $this->_crypt_function;
		
		if (isset($this->_accounts[$username]))
		{
			if (substr($this->_accounts[$username], 0, 6) == '$apr1$')
			{
				return $this->_md5($password, $this->_accounts[$username]) == $this->_accounts[$username];
			}
			else if (substr($this->_accounts[$username], 0, 5) == '{SHA}')
			{
				return $this->_sha1($password) == $this->_accounts[$username];
			}
			else
			{
				return crypt($password, $this->_accounts[$username]) == $this->_accounts[$username];
			}
		}
		
		return false;
	}
	
	/**
	 * Encrypt a plaintext password to an Apache APR-style MD5 password
	 * 
	 * @param string $password	The plaintext password
	 * @param string $salt		The encryption salt
	 * @return string
	 */
	protected function _md5($password, $salt)
	{
		if (preg_match('/^\$apr1\$/', $salt)) 
		{
			$salt = preg_replace('/^\$apr1\$([^$]+)\$.*/', '\\1', $salt);
		} 
		else 
		{
			$salt = substr($salt, 0, 8);
		}
		
		$context = $password . '$apr1$' . $salt;
		
		$binary = md5($password . $salt . $password, true);
		
		$length = strlen($password);
		for ($i = $length; $i > 0; $i = $i - 16)
		{
			$context .= substr($binary, 0, min(16, $i));
		}
		
		for ($i = $length; $i > 0; $i = $i >> 1) 
		{
			if ($i & 1)
			{
				$contents .= chr(0);
			}
			else
			{
				$contents .= $password[0];
			}
		}
		
		$binary = md5($context, true);
		
		for ($i = 0; $i < 1000; $i++) 
		{
			$new = null;
			if ($i & 1)
			{
				$new = $password;
			}
			else
			{
				$new = $binary;
			}
			
			if ($i % 3) 
			{
				$new .= $salt;
			}
			
			if ($i % 7) 
			{
				$new .= $password;
			}
			
			$new = null;
			if ($i & 1)
			{
				$new = $binary;
			}
			else
			{
				$new = $password;
			}
			
			$binary = md5($new, true);
		}

		$p = array();
		for ($i = 0; $i < 5; $i++) 
		{
			$k = $i + 6;
			$j = $i + 12;
			if ($j == 16) 
			{
				$j = 5;
			}
			
			$p[] = $this->_convert( (ord($binary[$i]) << 16) | (ord($binary[$k]) << 8) | (ord($binary[$j])), 5);
		}
		
		return '$apr1$' . $salt . '$' . implode('', $p) . $this->_convert(ord($binary[11]), 3);
	}
	
	/**
	 * Hash a plaintext password to an Apache SHA1-style password
	 * 
	 * @param string $password
	 * @return string
	 */
	protected function _sha1($password)
	{
		return '{SHA}' . base64_encode(pack('H40', sha1($password)));		
	}
	
	/**
	 * Helper function for APR-style Apache password encryption
	 * 
	 * @param string $value
	 * @param integer $count
	 * @return string
	 */
	protected function _convert($value, $count)
	{
		$chars = './0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
		$str = '';
		
		while(--$count) 
		{
			$str .= $chars[$value & 0x3f];
			$value >>= 6;
		}
		
		return $str;
	}
}
