<?php
namespace Ups;

use SimpleXMLElement;
use Exception;

class Request implements RequestInterface
{
    /**
     * @var string
     */
    protected $access;

    /**
     * @var string
     */
    protected $request;

    /**
     * @var string
     */
	protected $endpointUrl;
	
	private $request_cache_timeout = 600;//10 mins

    /**
     * Send request to UPS
     *
     * @param string $access The access request xml
     * @param string $request The request xml
     * @param string $endpointurl The UPS API Endpoint URL
     * @return ResponseInterface
     * @throws Exception
     * todo: make access, request and endpointurl nullable to make the testable
     */
    public function request($access, $request, $endpointurl)
    {
        $this->setAccess($access);
        $this->setRequest($request);
        $this->setEndpointUrl($endpointurl);

        // Create POST request
        $form = array(
            'http' => array(
                'method' => 'POST',
                'header' => 'Content-type: application/x-www-form-urlencoded',
                'content' => $this->getAccess() . $this->getRequest()
            )
		);
		
		//Check Cache
		$cache_file = ($cache_dir = $this->getRequestCacheDir()) ? $cache_dir.md5($form['http']['content']) : false;
		if(!$cache_file || !file_exists($cache_file) || !is_readable($cache_file) || time() - filemtime($cache_file) > $this->request_cache_timeout || !($response = file_get_contents($cache_file))){
			//No Cache Available.  Make new HTTP request
			$request = stream_context_create($form);
			if(!ini_get('allow_url_fopen'))
				throw new Exception("Failure: allow_url_fopen must be enabled (php.ini)");

			//Call fopen URL (with error reporting temporarily enabled)
			ob_start();
			$_error_reporting = ini_get('error_reporting');
			$_display_errors = ini_get('dislay_errors');
			$_html_errors = ini_get('html_errors');
			error_reporting(E_ALL);
			ini_set('display_errors', 1);#SAFE_DEBUG_ENABLED
			ini_set('html_errors', 0);
			$handle = fopen($this->getEndpointUrl(), 'rb', false, $request);
			error_reporting($_error_reporting);
			ini_set('display_errors', $_display_errors);#SAFE_DEBUG_ENABLED
			ini_set('html_errors', $_html_errors);
			$errs = ob_get_clean();

			if (!$handle){
				if(!ini_get('curl.cainfo')) $errs .= "\n Check php.ini value for \'curl.cainfo\'";
				if(!ini_get('openssl.cafile')) $errs .= "\n Check php.ini value for \'openssl.cafile\'";
				throw new Exception("Failure: Connection to Endpoint URL failed.".($errs ? ' '.$errs : ''));
			}
			$response = stream_get_contents($handle);
			fclose($handle);
			//Write To cache
			if($cache_file) @file_put_contents($cache_file, $response);
		}

        if ($response != false) {
            $text = $response;
            $response = new SimpleXMLElement($response);
            if (isset($response->Response) && isset($response->Response->ResponseStatusCode)) {
                $responseInstance = new Response;
                return $responseInstance->setText($text)->setResponse($response);
            }
        }

        throw new Exception("Failure: Response is invalid.");
	}

	public function getRequestCacheDir(){
		$cache_dir = NP_BASE_DIR . IMAGE_UPLOAD_DIR . 'temp' . DIRECTORY_SEPARATOR . 'upsrequest_cache' . DIRECTORY_SEPARATOR;
		if(!file_exists($cache_dir)){//Create Dir If Not Exists
			mkdir($cache_dir, 0777, true);
			if(!file_exists($cache_dir)) return false;//Still not existing?
		}
		//Cleanup Temp Dir
		if(rand(0, 256) == 1) cleanup_temp_directory($cache_dir, '', $this->request_cache_timeout / (24 * 60 * 60));
		return $cache_dir;
	}

    /**
     * @param $access
     * @return $this
     */
    public function setAccess($access)
    {
        $this->access = $access;
        return $this;
    }

    /**
     * @return string
     */
    public function getAccess()
    {
        return $this->access;
    }

    /**
     * @param $request
     * @return $this
     */
    public function setRequest($request)
    {
        $this->request = $request;
        return $this;
    }

    /**
     * @return string
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * @param $endpointUrl
     * @return $this
     */
    public function setEndpointUrl($endpointUrl)
    {
        $this->endpointUrl = $endpointUrl;
        return $this;
    }

    /**
     * @return string
     */
    public function getEndpointUrl()
    {
        return $this->endpointUrl;
    }
}