<?php

/**
 * Authenticate QuickBooks Web Connector sessions against a user-defined function
 * 
 * Your user-defined function should have a function header that looks 
 * something like this:
 * <code>
 * function my_auth_function($username, $password, &$company_file, &$wait_before_next_update, &$min_run_every_n_seconds, $arr_params)
 * {
 * 	...
 * }
 * </code>
 * 
 * Your function should return either TRUE for a successful login, or FALSE for 
 * an invalid login. 
 * 
 * The parameters are as follows:
 * 	- string $username		The username of the user trying to log in
 * 	- string $password		The password the user has supplied
 * 	- string $company_file	You can set this pass-by-reference parameter to the name of the company file if the login suceeeds
 * 	- integer $wait_before_next_update	You can set this pass-by-reference parameter to the number of seconds to wait before doing the next update
 * 	- integer $min_run_every_n_seconds	You can set this pass-by-reference parameter to the minimum number of seconds between QBWC connections
 * 	- array $arr_params		This will be set to an array of any other parameters you passed into the function via the DSN query string
 * 
 * @author Keith Palmer <keith@consolibyte.com>
 * @license LICENSE.txt
 * 
 * @package QuickBooks
 * @subpackage Authenticate
 */

/**
 * Authenticate interface
 */
require_once 'QuickBooks/Authenticate.php';

/**
 * Utilities for parsing DSN string
 */
require_once 'QuickBooks/Utilities.php';

/**
 * Authenticate against a user-defined function
 */
class QuickBooks_Authenticate_Function implements QuickBooks_Authenticate
{
	/**
	 * The custom authentication function you want to call
	 * @var string
	 */
	protected $_function;
	
	/**
	 * Parameters to the authentication function
	 * @var array 
	 */
	protected $_params;
	
	/**
	 * Create a new authentication handler
	 * 
	 * @param string $dsn	A DSN-style string indicating the function name and other parameters, i.e.: function://my_function_name?param1=value1&param2=value2
	 */
	public function __construct($dsn)
	{
		$conn_defaults = array(
			'scheme' => 'function', 
			'user' => '', 				// ignored
			'pass' => '',				// ignored
			'host' => 'your_function_name',  
			'port' => 0,				// ignored
			'path' => '',  
			'query' => '', 				// other parameters to the function 
			);
		
		// function://my_function_name?param1=value1&param2=value2
		$parse = QuickBooks_Utilities::parseDSN($dsn, $conn_defaults);
		
		$vars = array();
		parse_str($parse['query'], $vars);
		
		$this->_function = $parse['host'];
		$this->_params = array_merge($vars);
	}
	
	/**
	 * Authenticate a username and password against an 'htpasswd' file (generated by Apache) 
	 * 
	 * @param string $username
	 * @param string $password
	 * @param string $company_file				
	 * @param integer $wait_before_next_update	
	 * @param integer $min_run_every_n_seconds
	 * @return boolean
	 */
	public function authenticate($username, $password, &$company_file, &$wait_before_next_update, &$min_run_every_n_seconds)
	{
		if (function_exists($this->_function))
		{
			$func = $this->_function;
			return $func($username, $password, $company_file, $wait_before_next_update, $min_run_every_n_seconds, $this->_params);
		}
		
		return false;
	}
}
